<?php

function wecker_hole_alle(): array
{
    $sql = "
        SELECT *
        FROM wecker
        ORDER BY weckzeit
    ";

    return cyldb_select($GLOBALS['pdo'], $sql);
}

function wecker_hole_aktive(): array
{
    $sql = "
        SELECT *
        FROM wecker
        WHERE aktiv = :aktiv
        ORDER BY weckzeit
    ";

    return cyldb_select($GLOBALS['pdo'], $sql, [
        'aktiv' => 1
    ]);
}

function wecker_hole(int $id): ?array
{
    $sql = "
        SELECT *
        FROM wecker
        WHERE id = :id
    ";

    $rows = cyldb_select($GLOBALS['pdo'], $sql, [
        'id' => $id
    ]);

    return $rows[0] ?? null;
}

function wecker_hole_wochentage(int $weckerId): array
{
    $sql = "
        SELECT wochentag
        FROM wecker_wochentage
        WHERE wecker_id = :id
        ORDER BY wochentag
    ";

    $rows = cyldb_select($GLOBALS['pdo'], $sql, [
        'id' => $weckerId
    ]);

    return array_column($rows, 'wochentag');
}

function wecker_hole_audio(int $weckerId): array
{
    $sql = "
        SELECT id, datei, titel, position, sortierung
        FROM wecker_audio
        WHERE wecker_id = :id
        ORDER BY position, sortierung, id
    ";
    return cyldb_select($GLOBALS['pdo'], $sql, ['id' => $weckerId]);
}

function wecker_setze_audio(int $weckerId, array $items): void
{
    cyldb_exec(
        $GLOBALS['pdo'],
        'DELETE FROM wecker_audio WHERE wecker_id = :id',
        ['id' => $weckerId]
    );

    $sql = "
        INSERT INTO wecker_audio (wecker_id, datei, titel, position, sortierung)
        VALUES (:wecker_id, :datei, :titel, :position, :sortierung)
    ";
    $sort = ['vorher' => 0, 'gleichzeitig' => 0, 'danach' => 0];
    foreach ($items as $item) {
        $pos = $item['position'] ?? 'danach';
        if (!in_array($pos, ['vorher', 'gleichzeitig', 'danach'], true)) {
            $pos = 'danach';
        }
        $sort[$pos]++;
        cyldb_exec($GLOBALS['pdo'], $sql, [
            'wecker_id'  => $weckerId,
            'datei'      => $item['datei'] ?? '',
            'titel'      => !empty($item['titel']) ? $item['titel'] : null,
            'position'   => $pos,
            'sortierung' => $item['sortierung'] ?? $sort[$pos]
        ]);
    }
}

function wecker_hole_komplett(int $id): ?array
{
    $wecker = wecker_hole($id);

    if (!$wecker) {
        return null;
    }

    if ($wecker['wiederholung'] === 'woechentlich') {
        $wecker['wochentage'] = wecker_hole_wochentage($id);
    } else {
        $wecker['wochentage'] = [];
    }

    $wecker['audio'] = wecker_hole_audio($id);

    return $wecker;
}

function wecker_anlegen(array $daten): int
{
    $sql = "
        INSERT INTO wecker (
            weckzeit,
            weckdatum,
            wiederholung,
            sprachtext,
            aktiv
        ) VALUES (
            :zeit,
            :datum,
            :wiederholung,
            :text,
            :aktiv
        )
    ";

    return cyldb_exec($GLOBALS['pdo'], $sql, [
        'zeit'         => $daten['weckzeit'],
        'datum'        => $daten['weckdatum'],
        'wiederholung' => $daten['wiederholung'],
        'text'         => $daten['sprachtext'],
        'aktiv'        => $daten['aktiv'] ?? 1
    ]);
}

function wecker_setze_wochentage(int $weckerId, array $tage): void
{
    cyldb_exec(
        $GLOBALS['pdo'],
        'DELETE FROM wecker_wochentage WHERE wecker_id = :id',
        ['id' => $weckerId]
    );

    $sql = "
        INSERT INTO wecker_wochentage (wecker_id, wochentag)
        VALUES (:id, :tag)
    ";

    foreach ($tage as $tag) {
        cyldb_exec($GLOBALS['pdo'], $sql, [
            'id'  => $weckerId,
            'tag' => $tag
        ]);
    }
}

function wecker_aktualisieren(int $id, array $daten): int
{
    $sql = "
        UPDATE wecker SET
            weckzeit = :zeit,
            weckdatum = :datum,
            wiederholung = :wiederholung,
            sprachtext = :text,
            aktiv = :aktiv
        WHERE id = :id
    ";
    $n = cyldb_exec($GLOBALS['pdo'], $sql, [
        'id'         => $id,
        'zeit'       => $daten['weckzeit'],
        'datum'      => $daten['weckdatum'] ?: null,
        'wiederholung' => $daten['wiederholung'],
        'text'       => $daten['sprachtext'],
        'aktiv'      => isset($daten['aktiv']) ? (int)(bool)$daten['aktiv'] : 1
    ]);

    if ($n > 0 && ($daten['wiederholung'] ?? '') === 'woechentlich' && isset($daten['wochentage']) && is_array($daten['wochentage'])) {
        wecker_setze_wochentage($id, $daten['wochentage']);
    } elseif ($n > 0) {
        wecker_setze_wochentage($id, []);
    }

    return $n;
}

function wecker_setze_aktiv(int $id, bool $aktiv): int
{
    return cyldb_exec(
        $GLOBALS['pdo'],
        'UPDATE wecker SET aktiv = :aktiv WHERE id = :id',
        [
            'aktiv' => $aktiv ? 1 : 0,
            'id'    => $id
        ]
    );
}

function wecker_loeschen(int $id): int
{
    return cyldb_exec(
        $GLOBALS['pdo'],
        'DELETE FROM wecker WHERE id = :id',
        ['id' => $id]
    );
}

function einstellungen_hole_alle(): array
{
    $sql = "SELECT schluessel, wert FROM einstellungen";
    return cyldb_select($GLOBALS['pdo'], $sql);
}

function einstellung_hole(string $schluessel): ?string
{
    $rows = cyldb_select(
        $GLOBALS['pdo'],
        'SELECT wert FROM einstellungen WHERE schluessel = :k',
        ['k' => $schluessel]
    );

    return $rows[0]['wert'] ?? null;
}

function einstellung_setzen(string $schluessel, string $wert): int
{
    $sql = "
        INSERT INTO einstellungen (schluessel, wert) VALUES (:k, :v)
        ON DUPLICATE KEY UPDATE wert = VALUES(wert)
    ";
    return cyldb_exec($GLOBALS['pdo'], $sql, ['k' => $schluessel, 'v' => $wert]);
}

function tts_hole_aktive_platzhalter(): array
{
    $sql = "
        SELECT platzhalter, beschreibung
        FROM tts_platzhalter
        WHERE aktiv = 1
        ORDER BY sortierung, platzhalter
    ";

    return cyldb_select($GLOBALS['pdo'], $sql);
}

function tts_custom_hole_alle(): array
{
    $sql = "
        SELECT id, platzhalter, ersetzungstext, beschreibung, aktiv, sortierung
        FROM tts_custom
        ORDER BY sortierung, platzhalter
    ";
    return cyldb_select($GLOBALS['pdo'], $sql);
}

function tts_custom_hole_aktive(): array
{
    $sql = "
        SELECT platzhalter, ersetzungstext
        FROM tts_custom
        WHERE aktiv = 1
        ORDER BY sortierung, platzhalter
    ";
    return cyldb_select($GLOBALS['pdo'], $sql);
}

function tts_custom_hole(int $id): ?array
{
    $rows = cyldb_select(
        $GLOBALS['pdo'],
        'SELECT id, platzhalter, ersetzungstext, beschreibung, aktiv, sortierung FROM tts_custom WHERE id = :id',
        ['id' => $id]
    );
    return $rows[0] ?? null;
}

function tts_custom_anlegen(array $daten): int
{
    $sql = "
        INSERT INTO tts_custom (platzhalter, ersetzungstext, beschreibung, aktiv, sortierung)
        VALUES (:platzhalter, :ersetzungstext, :beschreibung, :aktiv, :sortierung)
    ";
    cyldb_exec($GLOBALS['pdo'], $sql, [
        'platzhalter'    => $daten['platzhalter'],
        'ersetzungstext' => $daten['ersetzungstext'],
        'beschreibung'   => $daten['beschreibung'] ?? null,
        'aktiv'         => isset($daten['aktiv']) ? (int)(bool)$daten['aktiv'] : 1,
        'sortierung'    => (int)($daten['sortierung'] ?? 0)
    ]);
    return (int)$GLOBALS['pdo']->lastInsertId();
}

function tts_custom_aktualisieren(int $id, array $daten): int
{
    $sql = "
        UPDATE tts_custom SET
            platzhalter = :platzhalter,
            ersetzungstext = :ersetzungstext,
            beschreibung = :beschreibung,
            aktiv = :aktiv,
            sortierung = :sortierung
        WHERE id = :id
    ";
    return cyldb_exec($GLOBALS['pdo'], $sql, [
        'id'             => $id,
        'platzhalter'    => $daten['platzhalter'],
        'ersetzungstext' => $daten['ersetzungstext'],
        'beschreibung'   => $daten['beschreibung'] ?? null,
        'aktiv'          => isset($daten['aktiv']) ? (int)(bool)$daten['aktiv'] : 1,
        'sortierung'     => (int)($daten['sortierung'] ?? 0)
    ]);
}

function tts_custom_loeschen(int $id): int
{
    return cyldb_exec(
        $GLOBALS['pdo'],
        'DELETE FROM tts_custom WHERE id = :id',
        ['id' => $id]
    );
}

/**
 * Lautstärke-Einstellungen für den Wecker-Service (TTS, Musik, Ducking).
 * Rückgabe: ['tts' => 0-100, 'musik' => 0-100, 'musik_ducking_prozent' => 0-100]
 * Während TTS abspielen: Musik auf (musik * musik_ducking_prozent / 100) setzen, danach wieder auf musik.
 */
function wecker_lautstaerke_holen(): array
{
    return [
        'tts'                   => (int) max(0, min(100, (int)(einstellung_hole('lautstaerke') ?: 100))),
        'musik'                 => (int) max(0, min(100, (int)(einstellung_hole('lautstaerke_musik') ?: 80))),
        'musik_ducking_prozent' => (int) max(0, min(100, (int)(einstellung_hole('musik_ducking_prozent') ?: 20))),
    ];
}

/**
 * Ersetzt TTS-Platzhalter im Text (für Wecker-Service beim Abspielen).
 * {{name}} = aus einstellungen tts_name, {{uhrzeit}}, {{datum}}, {{temperatur}}, plus tts_custom.
 */
function tts_text_ersetzen(string $text, ?DateTimeInterface $zeit = null): string
{
    if ($zeit === null) {
        $zeit = new DateTimeImmutable('now', new DateTimeZone(einstellung_hole('zeitzone') ?: 'Europe/Berlin'));
    }

    $ersetzungen = [
        '{{name}}'       => einstellung_hole('tts_name') ?: '',
        '{{uhrzeit}}'    => $zeit->format('H:i'),
        '{{datum}}'      => $zeit->format('d.m.Y'),
        '{{temperatur}}' => '', // später aus Sensor
    ];

    foreach (tts_custom_hole_aktive() as $row) {
        $ersetzungen[$row['platzhalter']] = $row['ersetzungstext'];
    }

    $out = $text;
    foreach ($ersetzungen as $platzhalter => $wert) {
        $out = str_replace($platzhalter, $wert, $out);
    }

    return $out;
}

