<?php
/**
 * Update-Funktion: Versionen vom Update-Server laden und installieren.
 * Nur nach Debug-Login erreichbar.
 */
require __DIR__ . '/config.php';
require __DIR__ . '/debug/auth.php';

requireLogin();

$updateServer = defined('UPDATE_SERVER') ? UPDATE_SERVER : 'https://update.piclock.cylone-network.com';
$installDir   = __DIR__;
$tmpZip       = __DIR__ . '/update.zip';
$backupDir    = __DIR__ . '/backup_' . date('YmdHis');

/** HTTP-Context mit Timeout für file_get_contents */
function httpContext($timeout = 30) {
    return stream_context_create([
        'http' => ['timeout' => $timeout],
        'ssl'  => ['verify_peer' => true]
    ]);
}

// ==============================
// VERSIONEN HOLEN (JSON)
// ==============================

if (isset($_GET['list'])) {
    $url = rtrim($updateServer, '/') . '/versions.php';
    $json = @file_get_contents($url, false, httpContext(15));
    header('Content-Type: application/json');
    echo $json !== false ? $json : json_encode(['error' => 'Versionen konnten nicht geladen werden']);
    exit;
}

// ==============================
// UPDATE AUSFÜHREN
// ==============================

if (isset($_GET['install'])) {
    header('Content-Type: text/plain; charset=utf-8');

    $version = $_GET['install'];
    if (!preg_match('/^[a-zA-Z0-9._-]+$/', $version)) {
        exit('Ungültige Version');
    }

    $url = rtrim($updateServer, '/') . '/versions.php';
    $json = @file_get_contents($url, false, httpContext(15));
    if ($json === false) {
        exit('Versionen konnten nicht geladen werden');
    }

    $versions = json_decode($json, true);
    if (!is_array($versions)) {
        exit('Ungültige Antwort vom Update-Server');
    }

    $selected = null;
    foreach ($versions as $v) {
        if (isset($v['version']) && $v['version'] === $version) {
            $selected = $v;
            break;
        }
    }

    if (!$selected || empty($selected['file'])) {
        exit('Version nicht gefunden');
    }

    $downloadUrl = rtrim($updateServer, '/') . '/releases/' . $selected['file'];
    echo "Download: $downloadUrl\n";
    flush();

    $zipData = @file_get_contents($downloadUrl, false, httpContext(120));
    if ($zipData === false || $zipData === '') {
        exit('Download fehlgeschlagen');
    }

    if (file_put_contents($tmpZip, $zipData) === false) {
        exit('ZIP konnte nicht gespeichert werden');
    }
    unset($zipData);

    if (!file_exists($tmpZip) || filesize($tmpZip) === 0) {
        exit('Download fehlgeschlagen');
    }

    echo "Backup...\n";
    flush();
    if (!@mkdir($backupDir, 0755, true)) {
        exit('Backup-Ordner konnte nicht erstellt werden');
    }

    if (PHP_OS_FAMILY !== 'Windows') {
        exec('cp -r ' . escapeshellarg($installDir) . '/* ' . escapeshellarg($backupDir) . ' 2>/dev/null');
    } else {
        foreach (scandir($installDir) as $f) {
            if ($f === '.' || $f === '..') continue;
            $src = $installDir . '/' . $f;
            $dst = $backupDir . '/' . $f;
            if (is_dir($src)) {
                @mkdir($dst);
            } else {
                @copy($src, $dst);
            }
        }
    }

    echo "Übernahme der Änderungen (bestehende Dateien werden überschrieben)...\n";
    flush();

    $zip = new ZipArchive();
    if ($zip->open($tmpZip) !== true) {
        exit('ZIP Fehler');
    }
    $zip->extractTo($installDir);
    $zip->close();
    @unlink($tmpZip);

    echo "UPDATE SUCCESS\n";
    exit;
}

// ==============================
// UI
// ==============================

$url = rtrim($updateServer, '/') . '/versions.php';
$json = @file_get_contents($url, false, httpContext(15));
$versions = is_string($json) ? json_decode($json, true) : [];
if (!is_array($versions)) {
    $versions = [];
}
?>
<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="utf-8">
    <title>Update</title>
    <style>
        body { font-family: system-ui, sans-serif; margin: 20px; background: #0f1115; color: #e6e6e6; }
        a { color: #4da3ff; }
        ul { list-style: none; padding: 0; }
        li { padding: 8px 0; border-bottom: 1px solid #2a2f3a; }
        .error { color: #ff6b6b; }
    </style>
</head>
<body>
<h1>Update-System</h1>
<p><a href="debug/">← Zurück zu Debug</a></p>

<?php if (empty($versions)): ?>
    <p class="error">Keine Versionen geladen. Prüfe UPDATE_SERVER in config.php und ob versions.php erreichbar ist.</p>
<?php else: ?>
<ul>
    <?php foreach ($versions as $v): ?>
        <li>
            Version: <strong><?= htmlspecialchars($v['version'] ?? '-') ?></strong>
            <?php if (!empty($v['commit'])): ?> | Commit: <?= htmlspecialchars($v['commit']) ?><?php endif; ?>
            | <a href="?install=<?= urlencode($v['version'] ?? '') ?>">Installieren</a>
        </li>
    <?php endforeach; ?>
</ul>
<?php endif; ?>
</body>
</html>
